/******************************************************************************
*  SCT_L - is used for PWM0 at SCT_OUT0
*  SCT_H - is used for PWM1 at SCT_OUT1
*
* Both PWM0/1 are center aligned PWM signals
*
*************************  SCT_L half  ****************************************
*
*       |       PWM0 Period       |                         |
*       |                         |                         |
*                 +-----+                   +-----+                   +-----+
*                 |     |                   |     |                   |     |      OUT0
*    ---+---------+     +---------+---------+     +---------+---------+     +----
*
*                 |     |                   |     |                   |
*                EV0   EV0                 EV0   EV0                 EV0
*
*  MATCH0_L used for PWM0 frequency period
*  EV0 - all states - on MATCH1_L - set OUT0 if up counting, clear OUT0 if down counting
*******************************************************************************
*
*************************  SCT_H half  ****************************************
*
*       |     PWM1 Period     |                     |
*       |                     |                     |
*            +-----------+         +-----------+          +-----------+
*            |           |         |           |          |           |   OUT1
*    ---+----+           +----+----+           +----+-----+           +--
*
*            |           |         |           |          |           |
*           EV1         EV1       EV1         EV1        EV1         EV1
*
*  MATCH0_H used for PWM1 frequency period
*  EV1 - all states - on MATCH1_H - set OUT1 if up counting, clear OUT1 if down counting
*
* Pins used in this application:
*
*	P0.14 [O] - SCT_OUT0 : PWM0
*	P0.15 [O] - SCT_OUT1 : PWM1
*
******************************************************************************/
#include "LPC8xx.h"

#define PWM0_FREQ       10000                               // PWM0 required frequency = 10KHz
#define PWM1_FREQ       600                                 // PWM1 required frequency = 600Hz

#define PWM0_PERIOD     (SystemCoreClock / (PWM0_FREQ * 2)) // PWM0 L counter period (*2 because of bi-dir mode)
                                                            // example 24MHz/10KHz*2 = 1200 SCT clocks
#define PWM1_PERIOD     (SystemCoreClock / (PWM1_FREQ * 2)) // PWM1 H counter period (*2 because of bi-dir mode)
                                                            // example 24MHz/600*2 = 20000 SCT clocks

#define OUT0            0                                   // SCT_OUT0 as PWM0 output
#define OUT1            1                                   // SCT_OUT1 as PWM1 output

#define PWM0            14                                  // PWM0 at port pin P0_14
#define PWM1            15                                  // PWM1 at port pin P0_15

void PWM0_set(uint8_t val)                                  // set PWM0 duty cycle (from 0 to 100%)
{
#define PWM0_STEP       (PWM0_PERIOD / 100)                 // PWM0 resolution in 100 steps

    if (val == 0)                                           // check val between 0% and 100%
       LPC_SCT->MATCHREL[1].L = 0;
    else if (val < 100)
       LPC_SCT->MATCHREL[1].L = (PWM0_STEP * val) - 1;
    else
       LPC_SCT->MATCHREL[1].L = PWM0_PERIOD - 2;            // set to 100% duty cycle
}

void PWM1_set(uint8_t val)                                  // set PWM1 duty cycle (from 0 to 100%)
{
#define PWM1_STEP       (PWM1_PERIOD / 100)                 // PWM1 resolution in 100 steps

    if (val == 0)                                           // check val between 0% and 100%
       LPC_SCT->MATCHREL[1].H = 0;
    else if (val < 100)
       LPC_SCT->MATCHREL[1].H = (PWM1_STEP * val) - 1;
    else
       LPC_SCT->MATCHREL[1].H = PWM1_PERIOD - 2;            // set to 100% duty cycle
}

void SCT_Init(void)
{
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 7) | (1 << 8);       // enable the SWM and SCT clock

    LPC_SWM->PINASSIGN6 &= ((PWM0 << 24) | 0x00FFFFFF);     // SCT_OUT0 = PWM0
    LPC_SWM->PINASSIGN7 &= ((PWM1 << 0)  | 0xFFFFFF00);     // SCT_OUT1 = PWM1

/********************************************************************
*  SCT_L: low part configuration:
********************************************************************/
                                              
    LPC_SCT->CONFIG           |= (1 << 17);                 // auto limit _L (on match 0)
    LPC_SCT->OUTPUT           |= (0 << OUT0);               // preset OUT0 low
//  LPC_SCT->OUTPUT           |= (1 << OUT0);               // preset OUT0 high (for low active signal)
    LPC_SCT->OUTPUTDIRCTRL    |= (0x1 << 0);                // reverse OUT0 set/clr when counter _L is
                                                            // down counting (center aligned mode)
    LPC_SCT->CTRL_L           |= (1 << 4);                  // bi-dir count mode

    LPC_SCT->MATCH[0].L        = PWM0_PERIOD - 1;           // match 0 @ PWM0 freq
    LPC_SCT->MATCHREL[0].L     = PWM0_PERIOD - 1;
    LPC_SCT->MATCH[1].L        = 0;                         // use match 1 for PWM0 duty cycle
    LPC_SCT->MATCHREL[1].L     = 0;                         // PWM0 off after power-up/reset 

    LPC_SCT->EVENT[0].STATE    = 0x00000003;                // event 0 happens in all states (both 0 and 1)
    LPC_SCT->EVENT[0].CTRL     = (1 << 0)  |                // MATCHSEL[3:0]   = related to match 1
                                 (0 << 4)  |                // HEVENT[4]       = event 0 belongs to the L timer
                                 (1 << 12) |                // COMBMODE[13:12] = match condition only
                                 (0 << 14) |                // STATELD[14]     = STATEV is added to state
                                 (0 << 15);                 // STATEV[15]      = 0 (no change)

    LPC_SCT->OUT[OUT0].SET     = (1 << 0);                  // event 0 sets the OUT0 signal
//  LPC_SCT->OUT[OUT0].CLR     = (1 << 0);                  // event 0 clears the OUT0 signal (for low active mode)

/********************************************************************
*  SCT_H: high part configuration:
********************************************************************/
                                              
    LPC_SCT->CONFIG           |= (1 << 18);                 // auto limit _H (on match 0)
    LPC_SCT->OUTPUT           |= (0 << OUT1);               // preset OUT1 low
//  LPC_SCT->OUTPUT           |= (1 << OUT1);               // preset OUT1 high (for low active signal)
    LPC_SCT->OUTPUTDIRCTRL    |= (0x2 << 2);                // reverse OUT0 set/clr when counter _H is
                                                            // down counting (center aligned mode)
    LPC_SCT->CTRL_H           |= (1 << 4);                  // bi-dir count mode

    LPC_SCT->MATCH[0].H        = PWM1_PERIOD - 1;           // match 0 @ PWM0 freq
    LPC_SCT->MATCHREL[0].H     = PWM1_PERIOD - 1;
    LPC_SCT->MATCH[1].H        = 0;                         // use match 1 for PWM1 duty cycle
    LPC_SCT->MATCHREL[1].H     = 0;                         // PWM1 off after power-up/reset 

    LPC_SCT->EVENT[1].STATE    = 0x00000003;                // event 1 happens in all states (both 0 and 1)
    LPC_SCT->EVENT[1].CTRL     = (1 << 0)  |                // MATCHSEL[3:0]   = related to match 1
                                 (1 << 4)  |                // HEVENT[4]       = event 1 belongs to the H timer
                                 (1 << 12) |                // COMBMODE[13:12] = match condition only
                                 (0 << 14) |                // STATELD[14]     = STATEV is added to state
                                 (0 << 15);                 // STATEV[15]      = 0 (no change)

    LPC_SCT->OUT[OUT1].SET     = (1 << 1);                  // event 1 sets the OUT1 signal
//  LPC_SCT->OUT[OUT1].CLR     = (1 << 1);                  // event 1 clears the OUT1 signal (for low active mode)

    LPC_SCT->CTRL_L           &= ~(1 << 2);                 // start the _L counter
    LPC_SCT->CTRL_H           &= ~(1 << 2);                 // start the _H counter
}
